module full_adder(x_in, y_in, c_in, s_out, c_out);
  input x_in, y_in, c_in;
  output s_out, c_out;
  wire w1, w2, w3;

  xor(w1, x_in, y_in);
  xor(s_out, w1, c_in);
  and(w2, w1, c_in);
  and(w3, x_in, y_in);
  or(c_out, w2, w3);
endmodule

module adder_subtractor(
  input [3:0] A,        // 4-bit input A
  input [3:0] B,        // 4-bit input B
  input mode,            // mode = 0 for addition, mode = 1 for subtraction
  output [7:0] result,   // 8-bit output result
  output carry_out	
);
  wire [3:0] B_xor, sum, c_out;

  // XOR B with mode for subtraction (when mode = 1, B is negated)
  xor(B_xor[0], B[0], mode);
  xor(B_xor[1], B[1], mode);
  xor(B_xor[2], B[2], mode);
  xor(B_xor[3], B[3], mode);

  // Perform addition with A and B_xor (B is modified based on mode)
  full_adder fa0(A[0], B_xor[0], mode, sum[0], c_out[0]);
  full_adder fa1(A[1], B_xor[1], c_out[0], sum[1], c_out[1]);
  full_adder fa2(A[2], B_xor[2], c_out[1], sum[2], c_out[2]);
  full_adder fa3(A[3], B_xor[3], c_out[2], sum[3], c_out[3]);

  // Final carry out (this will be the overflow in addition or borrow in subtraction)
  assign carry_out = c_out[3];


  assign result = sum;
endmodule

module array_multiplier(X, Y, P);
  input [3:0] X, Y;
  output [7:0] P;
  wire [3:0] s0_out, s1_out, s2_out, s3_out, s4_out;
  wire [3:0] c0_out, c1_out, c2_out, c3_out, c4_out;
  wire [3:0] x0y_in, x1y_in, x2y_in, x3y_in;

  // CSA row 0
  and (x0y_in[0], X[0], Y[0]);
  and (x0y_in[1], X[0], Y[1]);
  and (x0y_in[2], X[0], Y[2]);
  and (x0y_in[3], X[0], Y[3]);
  full_adder fa_x0y0 (x0y_in[0], 1'b0, 1'b0, s0_out[0], c0_out[0]);
  full_adder fa_x0y1 (x0y_in[1], 1'b0, 1'b0, s0_out[1], c0_out[1]);
  full_adder fa_x0y2 (x0y_in[2], 1'b0, 1'b0, s0_out[2], c0_out[2]);
  full_adder fa_x0y3 (x0y_in[3], 1'b0, 1'b0, s0_out[3], c0_out[3]);

  // CSA row 1
  and (x1y_in[0], X[1], Y[0]);
  and (x1y_in[1], X[1], Y[1]);
  and (x1y_in[2], X[1], Y[2]);
  and (x1y_in[3], X[1], Y[3]);
  full_adder fa_x1y0 (x1y_in[0], s0_out[1], c0_out[0], s1_out[0], c1_out[0]);
  full_adder fa_x1y1 (x1y_in[1], s0_out[2], c0_out[1], s1_out[1], c1_out[1]);
  full_adder fa_x1y2 (x1y_in[2], s0_out[3], c0_out[2], s1_out[2], c1_out[2]);
  full_adder fa_x1y3 (x1y_in[3], 1'b0, c0_out[3], s1_out[3], c1_out[3]);

  // CSA row 2
  and (x2y_in[0], X[2], Y[0]);
  and (x2y_in[1], X[2], Y[1]);
  and (x2y_in[2], X[2], Y[2]);
  and (x2y_in[3], X[2], Y[3]);
  full_adder fa_x2y0 (x2y_in[0], s1_out[1], c1_out[0], s2_out[0], c2_out[0]);
  full_adder fa_x2y1 (x2y_in[1], s1_out[2], c1_out[1], s2_out[1], c2_out[1]);
  full_adder fa_x2y2 (x2y_in[2], s1_out[3], c1_out[2], s2_out[2], c2_out[2]);
  full_adder fa_x2y3 (x2y_in[3], 1'b0, c1_out[3], s2_out[3], c2_out[3]);

  // CSA row 3
  and (x3y_in[0], X[3], Y[0]);
  and (x3y_in[1], X[3], Y[1]);
  and (x3y_in[2], X[3], Y[2]);
  and (x3y_in[3], X[3], Y[3]);
  full_adder fa_x3y0 (x3y_in[0], s2_out[1], c2_out[0], s3_out[0], c3_out[0]);
  full_adder fa_x3y1 (x3y_in[1], s2_out[2], c2_out[1], s3_out[1], c3_out[1]);
  full_adder fa_x3y2 (x3y_in[2], s2_out[3], c2_out[2], s3_out[2], c3_out[2]);
  full_adder fa_x3y3 (x3y_in[3], 1'b0, c2_out[3], s3_out[3], c3_out[3]);

  // CPA row 4
  full_adder fa_cp4(s3_out[1], c3_out[0], 1'b0, s4_out[0], c4_out[0]);
  full_adder fa_cp5(s3_out[2], c3_out[1], c4_out[0], s4_out[1], c4_out[1]);
  full_adder fa_cp6(s3_out[3], c3_out[2], c4_out[1], s4_out[2], c4_out[2]);
  full_adder fa_cp7(1'b0, c3_out[3], c4_out[2], s4_out[3], c4_out[3]);

  assign P = {s4_out[3], s4_out[2], s4_out[1], s4_out[0], s3_out[0], s2_out[0], s1_out[0], s0_out[0]};
endmodule


module comparator(
    input [3:0] A,    // 4-bit input A
    input [3:0] B,    // 4-bit input B
    output A_gt_B,     // Output: A > B
    output A_eq_B      // Output: A == B
);
    // Compare A and B
    assign A_gt_B = (A > B);  // A is greater than B
    assign A_eq_B = (A == B); // A is equal to B

endmodule



module ALU4_1(
    input [3:0] A,            // 4-bit input A
    input [3:0] B,            // 4-bit input B
    input [1:0] mode,         // 2-bit mode selector
    output reg [7:0] result,  // 8-bit result
    output reg carry_out,     // Carry-out (for addition/subtraction)
    output reg greater_than,  // Greater than flag (for comparison)
    output reg equal_to       // Equal to flag (for comparison)
);

    // Declare internal signals for instantiated modules
    wire [7:0] add_sub_result;
    wire add_sub_carry_out;
    wire [7:0] mul_result;
    wire [3:0] comp_result;
    wire g_than, e_to;	
    
    // Instantiating adder_subtractor module
    adder_subtractor adder_sub (
        .A(A), 
        .B(B), 
        .mode(mode[0]),  // Assuming 0 for addition, 1 for subtraction
        .result(add_sub_result),
        .carry_out(add_sub_carry_out)
    );

    // Instantiating multiplier module
    array_multiplier mul (
        .X(A), 
        .Y(B), 
        .P(mul_result)
    );
    
    // Instantiating comparator module
    comparator comp (
        .A(A), 
        .B(B), 
        .A_gt_B(g_than), 
        .A_eq_B(e_to)
    );
    
    // Combinational logic based on mode selection
    always @(*) begin
        case (mode)
            2'b00: begin // Addition
                result = add_sub_result;
                carry_out = add_sub_carry_out;
                greater_than = 0;
                equal_to = 0;
            end
            2'b01: begin // Subtraction
                result = add_sub_result;
                carry_out = add_sub_carry_out;
                greater_than = 0;
                equal_to = 0;
            end
            2'b10: begin // Multiplication
                result = mul_result;
                carry_out = 0;
                greater_than = 0;
                equal_to = 0;
            end
            2'b11: begin // Comparison
                result = 8'b0; // No result for comparison, just flags
                carry_out = 0;
		greater_than = g_than;
                equal_to = e_to;
            end
            default: begin
                result = 8'b0;
                carry_out = 0;
                greater_than = 0;
                equal_to = 0;
            end
        endcase
    end

endmodule
